"use client";
import { useState } from "react";
import Image from "next/image";
import styles from "./page.module.css";
import { Elements, PaymentElement, useStripe, useElements } from "@stripe/react-stripe-js";
import { loadStripe } from "@stripe/stripe-js";

const stripePromise = loadStripe('pk_live_51RMXCsKVjvPjpxCIbk87EaWEeg5jLHA16zbT0marelT6iZrLqg9UHJOkIlQQNxp0m62N2E3T8qjxygd0lfaIMUeK00ZHzj8Wia');

function ACHPaymentForm() {
  const stripe = useStripe();
  const elements = useElements();
  const [message, setMessage] = useState("");
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setMessage("");
    setLoading(true);
    try {
      if (!stripe || !elements) throw new Error("Stripe not loaded");
      const { error } = await stripe.confirmPayment({
        elements,
        confirmParams: {
          return_url: window.location.href,
        },
      });
      if (error) {
        setMessage(error.message || "Payment failed");
      } else {
        setMessage("Payment submitted! Please check your email for confirmation.");
      }
    } catch (err: unknown) {
      if (err instanceof Error) {
        setMessage(err.message || "Something went wrong");
      } else {
        setMessage("Something went wrong");
      }
    } finally {
      setLoading(false);
    }
  };

  return (
    <form className={styles.form} onSubmit={handleSubmit}>
      <div className={styles.stripeElementWrapper}>
        <PaymentElement options={{ paymentMethodOrder: ["us_bank_account"] }} />
      </div>
      <button type="submit" className={styles.submitBtn} disabled={loading}>
        {loading ? "Processing..." : "Pay via ACH"}
      </button>
      {message && <div className={styles.message}>{message}</div>}
    </form>
  );
}

export default function Home() {
  const [form, setForm] = useState({ name: "", email: "", amount: "", invoice: "", notes: "" });
  const [clientSecret, setClientSecret] = useState<string | null>(null);
  const [error, setError] = useState("");
  const [loading, setLoading] = useState(false);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    setForm({ ...form, [e.target.name]: e.target.value });
  };

  const handleInfoSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError("");
    setLoading(true);
    try {
      const res = await fetch("/api/create-ach-payment", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(form),
      });
      const data = await res.json();
      if (!res.ok) throw new Error(data.error || "Failed to create payment");
      setClientSecret(data.clientSecret);
    } catch (err: unknown) {
      if (err instanceof Error) {
        setError(err.message || "Something went wrong");
      } else {
        setError("Something went wrong");
      }
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className={styles.page}>
      <main className={styles.main}>
        <div className={styles.logoContainer}>
          <Image
            className={styles.logo}
            src="/dontafk-inverted.png"
            alt="Don't AFK Logo"
            width={400}
            height={80}
            priority
          />
        </div>
        {!clientSecret ? (
          <form className={styles.form} onSubmit={handleInfoSubmit}>
            <label>
              Name*
              <input type="text" name="name" required value={form.name} onChange={handleChange} />
            </label>
            <label>
              Email*
              <input type="email" name="email" required value={form.email} onChange={handleChange} />
            </label>
            <label>
              Amount (USD)*
              <input type="number" name="amount" min="1" step="0.01" required value={form.amount} onChange={handleChange} />
            </label>
            <label>
              Invoice #
              <input type="text" name="invoice" value={form.invoice} onChange={handleChange} />
            </label>
            <label>
              Notes
              <textarea name="notes" rows={2} value={form.notes} onChange={handleChange} />
            </label>
            <button type="submit" className={styles.submitBtn} disabled={loading}>
              {loading ? "Processing..." : "Next: Bank Info"}
            </button>
            {error && <div className={styles.message}>{error}</div>}
          </form>
        ) : (
          <Elements stripe={stripePromise} options={{ clientSecret }}>
            <ACHPaymentForm />
          </Elements>
        )}
      </main>
    </div>
  );
}
